function [Tr, Tr_all, Cens_all, Dist_all] = fTimeRecov(Seq, time)
% TIMERECOV calculates recovery time. This function uses an algorithm to 
% measure the recovery time (Tr) after a disturbance from sequential 
% spatial data (Seq). 
%
%   Input:
%       Seq     Sequential spatial data as a r * c * l matrix. The snapshot
%               per time step is a grid of r(ows) * c(olumns). Snapshots
%               are sequentially layered, with l(ayer) = number of 
%               snapshots.
%       time    a vector with a time stamp of each snapshot. Thus, the 
%               number of elements in time should be the same as l.
%   
%   Output:
%       Tr      Recovery time. If more than one recovery time is found the
%               average of the recovery time is given.
%               Tr > 0  : average recovery time, Tr
%               Tr = -1 : Never occupied
%               Tr = -2 : Stably occupied. No distrubances observed. 
%                         Always occupied in the sequence or initial 
%                         colonization occured.
%       Tr_all   All observed recovery times, Tr. Matrix of r * c * l-1
%       Cens_all All censored recovery times. Matrix of r * c * l-1
%       Dist_all All observed distrubances. Matrix of r * c * l-1
%
% Version 1.0 (25/07/2014)
% Version 1.1 (25/01/2017)
% Version 1.2 (08/03/2017)
% Coded and developed by Jim van Belzen
% published under the Creative Commons Attribution Non-Commercial license 
% which allows users to read, copy, distribute and make derivative works 
% for noncommercial purposes from the material, as long as the author of 
% the original work is cited.
% This code comes with no warranties or support
% http://people.zeelandnet.nl/jbelzen/

tic

%%- Check inputs
[r c l] = size(Seq);
[t(1) t(2)] = size(time);

if min(t) > 1,
    
    error('Input "time" should be a vector!')
    
elseif numel(unique(time)) ~= numel(time),
    
    error('Input "time" should contain all unique numbers in ascending order!')
    
elseif max(t) ~= l,
    
    error('Input "time" should be equal to the number of layers in "Seq"!')
    
end


%%- Initialize output variables
Tr_all = zeros(r,c,l-1);
Cens_all = zeros(r,c,l-1);
Dist_all = zeros(r,c,l-1);


%%- Calculate recovery times, Tr
for i = 2:l,
    
    % Initialize working variables
    dum = zeros(r,c);
    Tr = zeros(r,c);
        
    % Check if grid cells were disturbed
    Dist = (Seq(:,:,i-1) == 1 & Seq(:,:,i) == 0);
    
    % Check if grid cells recovered
    for j = i:l-1,
        
        Rec = (Dist == 1 & Seq(:,:,j+1) == 1 & dum == 0);
        
        % Tracking recovery time
        Tr = Tr + Rec * (time(j+1) - time(i));
        dum = dum + Rec;
        
    end
    
    % Determine if recovery is censored
    Tr = Tr + (Dist == 1 & dum == 0) * (time(end) - time(i));
    Cens = (Dist == 1 & dum == 0);
    
    % Store the measured data
    Tr_all(:,:,i-1) = Tr;       % Recovery time
    Cens_all(:,:,i-1) = Cens;   % Censored
    Dist_all(:,:,i-1) = Dist;   % Disturbances
    
end
  

%%- Calculate average recovery time, Tr
[row, col, lay] = size(Tr_all);
Tr = zeros(row,col);

for i = 1:row,
    
    for j = 1:col,
        
        dum = Tr_all(i,j,:);
        
        Tr(i,j) = mean(dum(dum~=0)); % mean of non-zero values
        
    end
    
end


%%- post-prepare output Tr
Tr(sum(Seq,3)==0) = -1; % Never occupied in sequence
Tr(sum(Seq,3)>0 & sum(Dist_all,3)==0) = -2; % Stably occupied.

toc
end